<?php
session_start();
require_once 'db_connection.php'; // provides $conn (mysqli) with utf8mb4

// Require login
if (!isset($_SESSION['user_id'])) { header('Location: login.php'); exit(); }

$user_role  = $_SESSION['role'] ?? '';
$user_id    = (int)($_SESSION['user_id'] ?? 0);
$branch_id  = (int)($_SESSION['branch_id'] ?? 0);
$username   = htmlspecialchars($_SESSION['username'] ?? 'User', ENT_QUOTES, 'UTF-8');

/* Role-scoped WHERE */
$where  = [];
$params = [];
$types  = "";

if ($user_role === 'branch_head') {
  $where[] = "l.branch_id = ?";
  $params[] = $branch_id; $types .= "i";
} elseif ($user_role === 'staff_user') {
  $where[] = "u.user_id = ?";
  $params[] = $user_id;   $types .= "i";
}
$whereSql = $where ? ("WHERE " . implode(" AND ", $where)) : "";

/* Query */
$sql = "
SELECT
  u.username AS staff_name,
  b.branch_name,
  COUNT(*) AS total_letters_received,
  SUM(CASE WHEN l.seen = 1 THEN 1 ELSE 0 END) AS seen_count,
  SUM(CASE WHEN l.seen = 0 THEN 1 ELSE 0 END) AS unseen_count,
  SUM(CASE WHEN l.action = 'no'  THEN 1 ELSE 0 END) AS no_action_count,
  SUM(CASE WHEN l.action = 'yes' THEN 1 ELSE 0 END) AS action_count,
  ROUND((SUM(CASE WHEN l.action = 'yes' THEN 1 ELSE 0 END)/NULLIF(COUNT(*),0))*100,2) AS archive_percentage
FROM letters l
JOIN users u    ON l.forward_to = u.user_id
JOIN branches b ON l.branch_id  = b.branch_id
{$whereSql}
GROUP BY u.username, b.branch_name
ORDER BY b.branch_name ASC, u.username ASC
";
$stmt = $conn->prepare($sql);
if ($types !== "") { $stmt->bind_param($types, ...$params); }
$stmt->execute();
$rs = $stmt->get_result();

/* Build rows + KPIs */
$rows = [];
$kpi  = ['total'=>0,'seen'=>0,'unseen'=>0,'action'=>0,'noAction'=>0];
if ($rs && $rs->num_rows) {
  while ($r = $rs->fetch_assoc()) {
    $rows[] = $r;
    $kpi['total']    += (int)$r['total_letters_received'];
    $kpi['seen']     += (int)$r['seen_count'];
    $kpi['unseen']   += (int)$r['unseen_count'];
    $kpi['action']   += (int)$r['action_count'];
    $kpi['noAction'] += (int)$r['no_action_count'];
  }
}
$stmt->close();
$conn->close();

/* Back link by role */
if     ($user_role==='super_admin')  $back = 'super_admin_dashboard.php';
elseif ($user_role==='letter_admin') $back = 'letter_admin_dashboard.php';
elseif ($user_role==='branch_head')  $back = 'branch_head_dashboard.php';
else                                  $back = 'staff_user_dashboard.php';
?>
<!DOCTYPE html>
<html lang="en" data-bs-theme="light">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1, viewport-fit=cover">
<title>Letters Statistics</title>

<link rel="preconnect" href="https://fonts.googleapis.com">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800&display=swap" rel="stylesheet">

<!-- Bootstrap 5 -->
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<!-- DataTables + Buttons -->
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css">
<link rel="stylesheet" href="https://cdn.datatables.net/buttons/3.0.2/css/buttons.bootstrap5.min.css">
<!-- Icons -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css"/>

<style>
:root{
  --brand:#7c3aed; --brand-2:#22d3ee; --ink:#0b1320; --bg:#f6f7fb; --surface:#ffffff;
  --muted:#6b7280; --radius:16px; --shadow: 0 16px 40px rgba(2,8,23,.08);
}
*{ box-sizing:border-box }
html,body{ height:100% }
body{ font-family:'Inter',system-ui,-apple-system,Segoe UI,Roboto,'Helvetica Neue',Arial; background:var(--bg); color:var(--ink) }

/* Top gradient bar */
.appbar{
  position: sticky; top:0; z-index:1030;
  background: linear-gradient(135deg,var(--brand),var(--brand-2));
  color:#fff; box-shadow:0 12px 28px rgba(124,58,237,.25);
  backdrop-filter: saturate(140%) blur(8px);
}
.appbar .avatar-badge{
  background: rgba(255,255,255,.18); border:1px solid rgba(255,255,255,.28);
}

/* Page shell */
.shell{ max-width: 1280px; margin-inline:auto; padding: 18px 14px; }

/* Cards */
.cardx{ background:var(--surface); border:1px solid rgba(2,8,23,.06); border-radius:var(--radius); box-shadow:var(--shadow) }
.cardx-header{ padding:14px 16px; border-bottom:1px solid rgba(2,8,23,.06); display:flex; align-items:center; justify-content:space-between }
.cardx-body{ padding:16px }

/* KPI Tiles */
.kpi{ display:flex; gap:14px; align-items:center }
.kpi .ico{ width:46px;height:46px;border-radius:12px;display:flex;align-items:center;justify-content:center;
  background:rgba(124,58,237,.12); color:#4c1d95 }
.kpi .value{ font-size:1.6rem; font-weight:800; line-height:1 }
.kpi .label{ color:var(--muted); font-weight:600; letter-spacing:.2px }
.progress{ height:6px; border-radius:999px; background:#eef2ff }
.progress-bar{ border-radius:999px; }

/* DataTables */
.table thead th{ white-space:nowrap }
.sticky-head thead th{ position:sticky; top:0; background:#f8fafc; z-index:2 }
.toolbar{ display:flex; flex-wrap:wrap; gap:.5rem; align-items:center }

/* Footer actions */
.footerbar{ position: sticky; bottom:0; z-index:10; margin-top:16px;
  background:linear-gradient(180deg,#fff,#f8f9ff); border:1px solid rgba(2,8,23,.06);
  border-radius:14px; padding:10px 14px; box-shadow:0 -8px 24px rgba(2,8,23,.06);
  display:flex; justify-content:space-between; align-items:center; gap:12px;
}

/* Dark mode */
@media (prefers-color-scheme: dark){ :root{ --bg:#0b1220; --surface:#0f1629; --ink:#e5e7eb } }
[data-bs-theme="dark"] .cardx{ background:#0f1629; border-color:rgba(255,255,255,.08) }
[data-bs-theme="dark"] .sticky-head thead th{ background:#0f1629 }
</style>
</head>
<body>

<!-- Top App Bar -->
<nav class="appbar">
  <div class="shell d-flex align-items-center justify-content-between">
    <div class="d-flex align-items-center gap-3">
      <img src="logo1.png" alt="" width="40" height="40" class="rounded"
           style="object-fit:cover;border:2px solid rgba(255,255,255,.35)" onerror="this.style.display='none'">
      <div>
        <div class="fw-bold fs-5">Letters Statistics</div>
        <small class="text-white-75">Valvettithurai Urban Council</small>
      </div>
    </div>
    <div class="d-flex align-items-center gap-2">
      <span class="badge avatar-badge">
        <i class="fa-solid fa-user-tie me-1"></i><?= $username; ?>
        <span class="mx-1">•</span><span class="text-capitalize"><?= htmlspecialchars($user_role,ENT_QUOTES,'UTF-8'); ?></span>
      </span>
      <a href="<?= $back; ?>" class="btn btn-outline-light btn-sm"><i class="fa-solid fa-arrow-left-long me-1"></i>Dashboard</a>
      <button id="themeToggle" class="btn btn-outline-light btn-sm" title="Toggle dark mode"><i class="fa-solid fa-moon"></i></button>
    </div>
  </div>
</nav>

<main class="shell">
  <!-- KPIs -->
  <div class="row g-3">
    <div class="col-12 col-md-6 col-xl-3">
      <div class="cardx p-3">
        <div class="kpi">
          <div class="ico"><i class="fa-solid fa-envelopes-bulk"></i></div>
          <div class="flex-grow-1">
            <div class="label">Total Letters</div>
            <div class="d-flex align-items-center justify-content-between">
              <div class="value"><?= number_format($kpi['total']); ?></div>
              <span class="badge bg-primary-subtle text-primary border border-primary-subtle">100%</span>
            </div>
            <div class="progress mt-2">
              <div class="progress-bar bg-primary" role="progressbar" style="width: 100%"></div>
            </div>
          </div>
        </div>
      </div>
    </div>

    <div class="col-12 col-md-6 col-xl-3">
      <div class="cardx p-3">
        <div class="kpi">
          <div class="ico"><i class="fa-solid fa-eye"></i></div>
          <div class="flex-grow-1">
            <div class="label">Seen</div>
            <div class="d-flex align-items-center justify-content-between">
              <div class="value"><?= number_format($kpi['seen']); ?></div>
              <?php $seenPct = $kpi['total'] ? round(($kpi['seen']/$kpi['total']) * 100, 1) : 0; ?>
              <span class="badge bg-success-subtle text-success border border-success-subtle"><?= $seenPct; ?>%</span>
            </div>
            <div class="progress mt-2">
              <div class="progress-bar bg-success" role="progressbar" style="width: <?= $seenPct; ?>%"></div>
            </div>
          </div>
        </div>
      </div>
    </div>

    <div class="col-12 col-md-6 col-xl-3">
      <div class="cardx p-3">
        <div class="kpi">
          <div class="ico"><i class="fa-solid fa-eye-slash"></i></div>
          <div class="flex-grow-1">
            <div class="label">Unseen</div>
            <div class="d-flex align-items-center justify-content-between">
              <div class="value"><?= number_format($kpi['unseen']); ?></div>
              <?php $unsPct = $kpi['total'] ? round(($kpi['unseen']/$kpi['total']) * 100, 1) : 0; ?>
              <span class="badge bg-warning-subtle text-warning border border-warning-subtle"><?= $unsPct; ?>%</span>
            </div>
            <div class="progress mt-2">
              <div class="progress-bar bg-warning" role="progressbar" style="width: <?= $unsPct; ?>%"></div>
            </div>
          </div>
        </div>
      </div>
    </div>

    <div class="col-12 col-md-6 col-xl-3">
      <div class="cardx p-3">
        <div class="kpi">
          <div class="ico"><i class="fa-solid fa-circle-check"></i></div>
          <div class="flex-grow-1">
            <div class="label">Action Taken</div>
            <div class="d-flex align-items-center justify-content-between">
              <div class="value"><?= number_format($kpi['action']); ?></div>
              <?php $actPct = $kpi['total'] ? round(($kpi['action']/$kpi['total']) * 100, 1) : 0; ?>
              <span class="badge bg-info-subtle text-info border border-info-subtle"><?= $actPct; ?>%</span>
            </div>
            <div class="progress mt-2">
              <div class="progress-bar bg-info" role="progressbar" style="width: <?= $actPct; ?>%"></div>
            </div>
          </div>
        </div>
      </div>
    </div>
  </div>

  <!-- Filters -->
  <div class="cardx mt-3">
    <div class="cardx-header">
      <div class="fw-bold"><i class="fa-solid fa-filter me-2"></i>Quick Filters</div>
      <div class="toolbar">
        <input id="quickSearch" class="form-control form-control-sm" placeholder="Global search…">
        <button id="resetFilters" class="btn btn-outline-secondary btn-sm"><i class="fa-solid fa-rotate"></i> Reset</button>
      </div>
    </div>
    <div class="cardx-body">
      <div class="row g-2">
        <div class="col-12 col-md-4">
          <label class="form-label small mb-1">Branch</label>
          <input id="filterBranch" type="text" class="form-control form-control-sm" placeholder="Filter by branch">
        </div>
        <div class="col-12 col-md-4">
          <label class="form-label small mb-1">Staff</label>
          <input id="filterStaff" type="text" class="form-control form-control-sm" placeholder="Filter by staff">
        </div>
        <div class="col-12 col-md-4">
          <label class="form-label small mb-1">Archive % ≥</label>
          <input id="filterPct" type="number" min="0" max="100" step="5" class="form-control form-control-sm" placeholder="e.g., 60">
        </div>
      </div>
    </div>
  </div>

  <!-- Table -->
  <div class="cardx mt-3">
    <div class="cardx-header">
      <div class="fw-bold"><i class="fa-solid fa-table me-2"></i>Staff & Branch Breakdown</div>
      <div id="dtButtons" class="toolbar"></div>
    </div>
    <div class="cardx-body">
      <?php if (empty($rows)): ?>
        <div class="alert alert-info mb-0"><i class="fa-solid fa-circle-info me-1"></i>No letters found for your scope.</div>
      <?php else: ?>
        <div class="table-responsive">
          <table id="statsTable" class="table table-hover align-middle sticky-head" style="width:100%">
            <thead class="table-light">
              <tr>
                <th>Staff</th>
                <th>Branch</th>
                <th class="text-center">Total</th>
                <th class="text-center">Seen</th>
                <th class="text-center">Unseen</th>
                <th class="text-center">No Action</th>
                <th class="text-center">Action</th>
                <th class="text-center">Archive %</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($rows as $r):
                $staff = htmlspecialchars($r['staff_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8');
                $branch= htmlspecialchars($r['branch_name'] ?? 'N/A',  ENT_QUOTES, 'UTF-8');
                $tot   = (int)$r['total_letters_received'];
                $seen  = (int)$r['seen_count'];
                $uns   = (int)$r['unseen_count'];
                $no    = (int)$r['no_action_count'];
                $act   = (int)$r['action_count'];
                $pct   = (float)$r['archive_percentage'];
                $initial = strtoupper(mb_substr($staff,0,1));
              ?>
              <tr>
                <td>
                  <div class="d-flex align-items-center gap-2">
                    <span class="rounded-circle d-inline-flex align-items-center justify-content-center"
                          style="width:34px;height:34px;background:rgba(124,58,237,.12);color:#7c3aed;font-weight:800;">
                      <?= $initial; ?>
                    </span>
                    <span class="fw-semibold"><?= $staff; ?></span>
                  </div>
                </td>
                <td><?= $branch; ?></td>
                <td class="text-center"><?= number_format($tot); ?></td>
                <td class="text-center"><?= number_format($seen); ?></td>
                <td class="text-center"><?= number_format($uns); ?></td>
                <td class="text-center"><?= number_format($no); ?></td>
                <td class="text-center"><?= number_format($act); ?></td>
                <td class="text-center">
                  <?php $cls = $pct>=80 ? 'success' : ($pct>=50 ? 'warning' : 'danger'); ?>
                  <span class="badge text-bg-<?= $cls; ?>"><?= number_format($pct,2); ?>%</span>
                </td>
              </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      <?php endif; ?>
    </div>
  </div>

  <!-- Footer actions -->
  <div class="footerbar">
    <div class="d-flex align-items-center gap-2">
      <a href="<?= $back; ?>" class="btn btn-outline-secondary btn-sm">
        <i class="fa-solid fa-arrow-left-long me-1"></i>Back to Dashboard
      </a>
    </div>
    <div class="d-flex align-items-center gap-2">
      <button class="btn btn-primary btn-sm" onclick="window.print()"><i class="fa-solid fa-print me-1"></i>Print</button>
      <button id="scrollTop" class="btn btn-outline-primary btn-sm" type="button" title="Top"><i class="fa-solid fa-arrow-up"></i></button>
    </div>
  </div>
</main>

<!-- JS -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>

<!-- DataTables -->
<script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>
<!-- Buttons -->
<script src="https://cdn.datatables.net/buttons/3.0.2/js/dataTables.buttons.min.js"></script>
<script src="https://cdn.datatables.net/buttons/3.0.2/js/buttons.bootstrap5.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.2.7/pdfmake.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.2.7/vfs_fonts.js"></script>
<script src="https://cdn.datatables.net/buttons/3.0.2/js/buttons.html5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/3.0.2/js/buttons.print.min.js"></script>
<script src="https://cdn.datatables.net/buttons/3.0.2/js/buttons.colVis.min.js"></script>

<script>
  // Theme toggle (persist)
  (function(){
    const btn = document.getElementById('themeToggle');
    const root = document.documentElement;
    const saved = localStorage.getItem('theme');
    if (saved) {
      root.setAttribute('data-bs-theme', saved);
    } else if (matchMedia('(prefers-color-scheme: dark)').matches) {
      root.setAttribute('data-bs-theme', 'dark');
    }
    btn?.addEventListener('click', ()=>{
      const cur = root.getAttribute('data-bs-theme') === 'dark' ? 'light' : 'dark';
      root.setAttribute('data-bs-theme', cur);
      localStorage.setItem('theme', cur);
    });
  })();

  // DataTable + buttons
  let dt;
  $(function(){
    const title = 'Letters Statistics (' + new Date().toISOString().slice(0,10) + ')';
    dt = $('#statsTable').DataTable({
      pageLength: 25,
      order: [[1,'asc'],[0,'asc']],
      autoWidth: false,
      stateSave: true,
      dom: 't', // custom toolbar & buttons appended manually
      columnDefs: [{ targets:[2,3,4,5,6,7], className:'text-center' }]
    });

    new $.fn.dataTable.Buttons(dt, {
      buttons: [
        { extend:'colvis',     className:'btn btn-outline-primary btn-sm', text:'Columns' },
        { extend:'copyHtml5',  className:'btn btn-outline-primary btn-sm', title },
        { extend:'csvHtml5',   className:'btn btn-outline-primary btn-sm', title, filename:'letters_stats' },
        { extend:'excelHtml5', className:'btn btn-outline-primary btn-sm', title, filename:'letters_stats' },
        { extend:'pdfHtml5',   className:'btn btn-outline-primary btn-sm', title, orientation:'landscape', pageSize:'A4' },
        { extend:'print',      className:'btn btn-outline-primary btn-sm', title }
      ]
    });
    dt.buttons().container().appendTo('#dtButtons');

    // Global quick search
    $('#quickSearch').on('input', function(){ dt.search(this.value).draw(); });

    // Column filters
    $('#filterBranch').on('input', function(){ dt.column(1).search(this.value).draw(); });
    $('#filterStaff').on('input',  function(){ dt.column(0).search(this.value).draw(); });
    $('#filterPct').on('input',    function(){
      const val = this.value;
      dt.column(7).search(val ? '^' + val : '', true, false).draw();
    });

    // Reset
    $('#resetFilters').on('click', function(){
      $('#quickSearch, #filterBranch, #filterStaff, #filterPct').val('');
      dt.search('').columns().search('').draw();
    });

    // Scroll top
    document.getElementById('scrollTop').addEventListener('click', ()=>window.scrollTo({top:0, behavior:'smooth'}));
  });
</script>
</body>
</html>
