<?php
// Start session & DB
session_start();
require_once 'db_connection.php';

// Auth: Staff User only
if (!isset($_SESSION['user_id']) || ($_SESSION['role'] ?? '') !== 'staff_user') {
    header("Location: login.php");
    exit();
}

$user_id = (int)($_SESSION['user_id'] ?? 0);

// Handle update request for action, description, and file_id
if (isset($_POST['update_action_description'])) {
    $letter_id   = (int)($_POST['letter_id'] ?? 0);
    $action_val  = ($_POST['action'] ?? '0') === '1' ? 'Yes' : 'No'; // store as Yes/No string
    $description = trim($_POST['description'] ?? '');
    $file_id     = (int)($_POST['file_id'] ?? 0);

    $sql = "UPDATE letters SET action = ?, description = ?, file_id = ? 
            WHERE letter_id = ? AND forward_to = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("ssiii", $action_val, $description, $file_id, $letter_id, $user_id);
        if ($stmt->execute()) {
            // PRG pattern
            header("Location: staff_seen_letters.php");
            exit();
        } else {
            $error_message = "Error updating letter: " . $stmt->error;
        }
        $stmt->close();
    } else {
        $error_message = "Failed to prepare statement.";
    }
}

// Fetch letters: seen=1, action='No', description empty
$letters = [];
$sqlList = "SELECT letter_id, letter_number, letter_date, department, subject, hod_comment, 
                   received_date, scan_copy, action, description, file_id
            FROM letters
            WHERE forward_to = ? AND seen = 1 AND action = 'No'
              AND (description IS NULL OR description = '')
            ORDER BY letter_date DESC";
if ($stmt = $conn->prepare($sqlList)) {
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $res = $stmt->get_result();
    while ($row = $res->fetch_assoc()) { $letters[] = $row; }
    $stmt->close();
} else {
    $error_message = "Error fetching letters.";
}

// Fetch files for this user (once, into array)
$files = [];
$file_sql = "SELECT file_id, file_name FROM file_management WHERE user_id = ? ORDER BY file_name ASC";
if ($fs = $conn->prepare($file_sql)) {
    $fs->bind_param("i", $user_id);
    $fs->execute();
    $fr = $fs->get_result();
    while ($f = $fr->fetch_assoc()) { $files[] = $f; }
    $fs->close();
}

$conn->close();

function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
?>
<!DOCTYPE html>
<html lang="en" data-bs-theme="light">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
  <title>Seen Letters Pending Action • Staff</title>

  <!-- Bootstrap 5.3 & Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
  <!-- Simple-DataTables (no jQuery) -->
  <link href="https://cdn.jsdelivr.net/npm/simple-datatables@9.0.0/dist/style.css" rel="stylesheet">

  <style>
    :root{
      --brand:#155eef; --border:#e8ecf7; --surface:#ffffff; --radius:14px;
    }
    body{ background:#f6f8ff; }
    .page-wrap{ max-width: 1280px; }
    .page-card{ background:var(--surface); border:1px solid var(--border); border-radius:var(--radius); }
    thead th{ background:#f8faff; position:sticky; top:0; z-index:1; }
    .scan-link{ max-width:240px; display:inline-block; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; }
    .btn-soft{ background:#eef3ff; border:1px solid #dbe6ff; color:#155eef; }
    .btn-soft:hover{ background:#e3ebff; color:#0e46d9; }
    .badge-soft-danger{ background:#fff3f3; color:#b42318; border:1px solid #ffd5d5; }
    .form-mini .form-control, .form-mini .form-select, .form-mini .btn{ padding:.4rem .6rem; font-size:.9rem; }
    .datatable-table > thead > tr > th { border-bottom: 1px solid var(--border); }
    .datatable-wrapper .datatable-top{ padding:0 0 1rem 0; }
    .datatable-wrapper .datatable-bottom{ padding:1rem 0 0 0; }
  </style>
</head>
<body>

  <!-- Header -->
  <header class="border-bottom bg-white">
    <div class="container page-wrap d-flex align-items-center justify-content-between py-3">
      <div class="d-flex align-items-center gap-2">
        <a href="staff_user_dashboard.php" class="btn btn-outline-secondary btn-sm"><i class="bi bi-arrow-left"></i></a>
        <div>
          <h5 class="mb-0">Seen Letters Pending Action</h5>
          <small class="text-secondary">Seen = Yes • Action = No • Description empty</small>
        </div>
      </div>
      <img src="logo.jpg" alt="Logo" width="44" height="44" class="rounded">
    </div>
  </header>

  <main class="container page-wrap my-4">

    <?php if (!empty($error_message)): ?>
      <div class="alert alert-danger d-flex align-items-center" role="alert">
        <i class="bi bi-exclamation-triangle-fill me-2"></i>
        <div><?= h($error_message) ?></div>
      </div>
    <?php endif; ?>

    <div class="page-card p-3 p-md-4">
      <div class="d-flex align-items-center justify-content-between mb-3">
        <div class="d-flex align-items-center gap-2">
          <span class="badge text-bg-primary"><i class="bi bi-envelope-check me-1"></i> Ready to action</span>
          <span class="text-secondary small">Total: <strong><?= count($letters) ?></strong></span>
        </div>
        <div class="d-flex gap-2">
          <a href="staff_user_dashboard.php" class="btn btn-outline-secondary btn-sm">
            <i class="bi bi-grid me-1"></i> Dashboard
          </a>
          <a href="view_letters_statistics.php" class="btn btn-soft btn-sm">
            <i class="bi bi-pie-chart me-1"></i> Statistics
          </a>
        </div>
      </div>

      <?php if (!count($letters)): ?>
        <div class="text-center py-5">
          <i class="bi bi-inboxes fs-1 text-secondary d-block mb-2"></i>
          <h6 class="mb-1">No items need your action</h6>
          <p class="text-secondary mb-3">All seen letters either have actions or descriptions already.</p>
          <a href="staff_user_dashboard.php" class="btn btn-primary btn-sm"><i class="bi bi-arrow-left-short"></i> Back to Dashboard</a>
        </div>
      <?php else: ?>
        <div class="table-responsive">
          <table id="lettersTable" class="table table-hover align-middle mb-0">
            <thead>
              <tr>
                <th>Letter No.</th>
                <th>Letter Date</th>
                <th>Department</th>
                <th>Subject</th>
                <th>HOD Comment</th>
                <th>Received</th>
                <th>Scan</th>
                <th>Update</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($letters as $row): ?>
                <tr id="row_<?= (int)$row['letter_id'] ?>">
                  <td><?= h($row['letter_number']) ?></td>
                  <td><?= h($row['letter_date']) ?></td>
                  <td><?= h($row['department']) ?></td>
                  <td class="text-truncate" style="max-width:360px;" title="<?= h($row['subject']) ?>"><?= h($row['subject']) ?></td>
                  <td class="text-truncate" style="max-width:260px;" title="<?= h($row['hod_comment']) ?>"><?= h($row['hod_comment']) ?></td>
                  <td><?= $row['received_date'] ? h($row['received_date']) : 'N/A' ?></td>
                  <td>
                    <?php if (!empty($row['scan_copy'])): ?>
                      <a class="scan-link" href="<?= h($row['scan_copy']) ?>" target="_blank" rel="noopener">
                        <i class="bi bi-box-arrow-up-right me-1"></i>View
                      </a>
                    <?php else: ?>
                      <span class="text-secondary">No file</span>
                    <?php endif; ?>
                  </td>
                  <td>
                    <!-- Inline update form -->
                    <form method="POST" class="form-mini d-flex flex-column gap-1 update-form" data-letter-no="<?= h($row['letter_number']) ?>">
                      <input type="hidden" name="letter_id" value="<?= (int)$row['letter_id'] ?>">
                      <div class="row g-1">
                        <div class="col-12 col-md-3">
                          <select name="action" class="form-select" required>
                            <option value="0" <?= ($row['action'] === 'No' ? 'selected':'') ?>>No</option>
                            <option value="1">Yes</option>
                          </select>
                        </div>
                        <div class="col-12 col-md-5">
                          <textarea name="description" class="form-control" rows="1" placeholder="Describe your action..." required><?= h($row['description']) ?></textarea>
                        </div>
                        <div class="col-12 col-md-4">
                          <select name="file_id" class="form-select" required>
                            <option value="">Select File</option>
                            <?php foreach ($files as $f): ?>
                              <option value="<?= (int)$f['file_id'] ?>" <?= ((int)$row['file_id'] === (int)$f['file_id'] ? 'selected':'') ?>>
                                <?= h($f['file_name']) ?>
                              </option>
                            <?php endforeach; ?>
                          </select>
                        </div>
                      </div>
                      <div class="d-flex justify-content-end">
                        <button type="button" class="btn btn-primary btn-sm mt-2 trigger-confirm">
                          <i class="bi bi-check2-circle me-1"></i> Update
                        </button>
                        <input type="hidden" name="update_action_description" value="1">
                      </div>
                    </form>
                    <div class="mt-1">
                      <span class="badge badge-soft-danger">Action: No</span>
                    </div>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      <?php endif; ?>
    </div>
  </main>

  <!-- Confirm Modal -->
  <div class="modal fade" id="confirmModal" tabindex="-1" aria-labelledby="confirmModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
      <div class="modal-content">
        <div class="modal-header border-0">
          <h6 class="modal-title" id="confirmModalLabel"><i class="bi bi-question-circle me-2"></i>Confirm Update</h6>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
        </div>
        <div class="modal-body">
          Are you sure you want to update action, description, and file for <strong id="refLetter">this letter</strong>?
        </div>
        <div class="modal-footer border-0">
          <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
          <button type="button" class="btn btn-primary" id="confirmBtn"><i class="bi bi-check2 me-1"></i>Yes, update</button>
        </div>
      </div>
    </div>
  </div>

  <!-- JS -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/simple-datatables@9.0.0" ></script>
  <script>
    // Simple-DataTables init
    (function(){
      const tbl = document.getElementById('lettersTable');
      if (tbl) {
        new simpleDatatables.DataTable(tbl, {
          searchable: true,
          fixedHeight: false,
          perPage: 10,
          perPageSelect: [10, 20, 50, 100],
          labels: {
            placeholder: "Search...",
            perPage: "{select} per page",
            noRows: "No entries to show",
            info: "Showing {start}–{end} of {rows} entries"
          }
        });
      }
    })();

    // Confirm modal wiring (no window.confirm)
    (function(){
      const modalEl = document.getElementById('confirmModal');
      const refEl   = document.getElementById('refLetter');
      const okBtn   = document.getElementById('confirmBtn');
      let pendingForm = null;

      document.querySelectorAll('.trigger-confirm').forEach(btn => {
        btn.addEventListener('click', function(){
          pendingForm = this.closest('form');
          const ref = pendingForm?.dataset?.letterNo || 'this letter';
          refEl.textContent = ref;
          const m = new bootstrap.Modal(modalEl);
          m.show();
        });
      });

      okBtn.addEventListener('click', function(){
        if (pendingForm) { pendingForm.submit(); }
      });
    })();
  </script>
</body>
</html>
