<?php
// Start the session and include the database connection
session_start();
require_once 'db_connection.php';  // Include your database connection file

// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");  // Redirect to login page if not logged in
    exit();
}

// Session values
$user_role  = $_SESSION['role'] ?? '';
$user_id    = (int)($_SESSION['user_id'] ?? 0);
$branch_id  = (int)($_SESSION['branch_id'] ?? 0);

// ---- Build role-safe SQL with prepared statements ----
$where  = [];
$params = [];
$types  = '';

if ($user_role === 'branch_head') {
    $where[]  = 'l.branch_id = ?';
    $params[] = $branch_id;
    $types   .= 'i';
} elseif ($user_role === 'staff_user') {
    $where[]  = 'u.user_id = ?';
    $params[] = $user_id;
    $types   .= 'i';
}
// super_admin / letter_admin -> no extra filters

$whereSql = $where ? 'WHERE '.implode(' AND ', $where) : '';

$sql = "
SELECT
  u.user_id,
  u.username,
  l.branch_id,
  b.branch_name,
  COUNT(l.letter_id) AS letter_count
FROM letters l
JOIN users u    ON l.user_id   = u.user_id
JOIN branches b ON l.branch_id = b.branch_id
{$whereSql}
GROUP BY u.user_id, u.username, l.branch_id, b.branch_name
ORDER BY b.branch_name, u.username
";

$stmt = $conn->prepare($sql);
if (!$stmt) { die('Prepare failed: '.$conn->error); }
if ($types !== '') { $stmt->bind_param($types, ...$params); }
$stmt->execute();
$result = $stmt->get_result();
if ($result === false) { die('Query failed: '.$conn->error); }

// Decide dashboard back link (UI-only)
if     ($user_role === 'super_admin')  { $back = 'super_admin_dashboard.php'; }
elseif ($user_role === 'letter_admin') { $back = 'letter_admin_dashboard.php'; }
elseif ($user_role === 'branch_head')  { $back = 'branch_head_dashboard.php'; }
else                                   { $back = 'staff_user_dashboard.php'; }
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
  <title>Branch Head Wise Letters</title>

  <!-- Bootstrap 4 -->
  <link rel="stylesheet" href="https://maxcdn.bootstrapcdn.com/bootstrap/4.0.0/css/bootstrap.min.css"/>
  <!-- DataTables CSS -->
  <link rel="stylesheet" href="https://cdn.datatables.net/1.10.21/css/jquery.dataTables.min.css"/>
  <!-- Font Awesome -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css"/>

  <style>
    :root{
      --brand:#2563eb; --brand-2:#60a5fa; --bg:#f4f7fb; --ink:#0b1320;
      --card-r:16px; --soft-shadow:0 16px 32px rgba(2,8,23,.07);
    }
    body{
      background:
        radial-gradient(70rem 70rem at 120% -20%, rgba(37,99,235,.08), transparent 60%),
        radial-gradient(70rem 70rem at -20% 120%, rgba(37,99,235,.06), transparent 60%),
        var(--bg);
      color:var(--ink);
      font-family: system-ui, -apple-system, Segoe UI, Roboto, "Helvetica Neue", Arial;
    }
    .topbar{
      position: sticky; top: 0; z-index: 1030;
      background: linear-gradient(135deg, var(--brand), var(--brand-2));
      color: #fff;
      box-shadow: 0 10px 24px rgba(37,99,235,.25);
      padding: .75rem 0;
    }
    .brand-title{ font-weight:800; letter-spacing:.2px; }
    .avatar-badge{
      background: rgba(255,255,255,.18);
      border: 1px solid rgba(255,255,255,.28);
      color:#fff;
    }
    .shell{ max-width: 1250px; margin: 24px auto; padding: 0 12px; }
    .panel{ background: #fff; border: 1px solid rgba(2,8,23,.06); border-radius: var(--card-r); box-shadow: var(--soft-shadow); }
    .panel-header{ display:flex; align-items:center; justify-content:space-between; gap:1rem; padding: 1rem 1.25rem; border-bottom:1px solid rgba(2,8,23,.06); }
    .kpi{ background:#fff; border:1px solid rgba(2,8,23,.06); border-radius:14px; padding:14px 16px; box-shadow: var(--soft-shadow); display:flex; align-items:center; gap:12px; }
    .kpi .icon{ width:44px; height:44px; border-radius:10px; display:flex; align-items:center; justify-content:center; background: rgba(37,99,235,.12); color: var(--brand); }
    .kpi .value{ font-size:1.2rem; font-weight:800; line-height:1; }
    .kpi .label{ color:#6b7280; font-weight:600; margin-top:2px; }
    .table thead th{ background: #f8fafc; border-bottom: 1px solid rgba(2,8,23,.08); }
    .table-hover tbody tr:hover{ background-color:#f3f6ff; }
    .sticky-head thead th{ position: sticky; top: 0; z-index: 1; }
    .modal-header{ background: var(--brand); color:#fff; border-bottom:0; }
    .modal-content{ border-radius: 14px; border: 1px solid rgba(2,8,23,.06); box-shadow: var(--soft-shadow); }
    .btn-primary{ background: var(--brand); border-color: var(--brand); }
    .btn-primary:hover{ background:#1e4fc7; border-color:#1e4fc7; }
    .btn-outline-light{ color:#fff; border-color:#fff; }
    .btn-outline-light:hover{ color:var(--brand); background:#fff; }
  </style>
</head>
<body>

<!-- Topbar -->
<div class="topbar">
  <div class="container d-flex align-items-center justify-content-between">
    <div class="d-flex align-items-center gap-3">
      <img src="logo1.png" width="38" height="38" class="rounded"
           style="object-fit:cover;border:2px solid rgba(255,255,255,.4)"
           onerror="this.style.display='none'" alt="Logo">
      <div>
        <div class="brand-title">Branch Head Wise Letters</div>
        <small class="text-white-50">Valvettithurai Urban Council</small>
      </div>
    </div>
    <div class="d-flex align-items-center">
      <span class="badge avatar-badge mr-2">
        <i class="fa fa-user-tie mr-1"></i><?php echo htmlspecialchars($_SESSION['username']??'User'); ?>
        <span class="mx-1">•</span><?php echo htmlspecialchars($user_role); ?>
      </span>
      <a href="<?php echo htmlspecialchars($back, ENT_QUOTES, 'UTF-8'); ?>"
         class="btn btn-outline-light btn-sm">
         <i class="fa fa-arrow-left mr-1"></i> Back to Dashboard
      </a>
    </div>
  </div>
</div>

<div class="shell">
  <div class="row">
    <div class="col-12 col-md-4 mb-3">
      <div class="kpi">
        <div class="icon"><i class="fa fa-users"></i></div>
        <div>
          <div class="value"><?php echo number_format($result->num_rows); ?></div>
          <div class="label">User / Branch pairs</div>
        </div>
      </div>
    </div>
    <div class="col-12 col-md-8 mb-3">
      <div class="kpi">
        <div class="icon"><i class="fa fa-lightbulb"></i></div>
        <div>
          <div class="value">Tip</div>
          <div class="label">Click “View Details” to open letters in a modal</div>
        </div>
      </div>
    </div>
  </div>

  <div class="panel">
    <div class="panel-header">
      <h5 class="mb-0"><i class="fa fa-table mr-2"></i>Users & Branch Summary</h5>
    </div>
    <div class="p-3 p-md-4">
      <?php if ($result->num_rows === 0): ?>
        <div class="alert alert-info mb-0">No users found.</div>
      <?php else: ?>
        <div class="table-responsive">
          <table id="usersTable" class="table table-hover table-bordered align-middle sticky-head" style="width:100%">
            <thead>
              <tr>
                <th><i class="fas fa-user"></i> User Name</th>
                <th><i class="fas fa-building"></i> Branch Name</th>
                <th class="text-center"><i class="fas fa-envelope"></i> Letter Count</th>
                <th class="text-center"><i class="fas fa-info-circle"></i> Action</th>
              </tr>
            </thead>
            <tbody>
              <?php while ($row = $result->fetch_assoc()):
                $uid   = (int)$row['user_id'];
                $uname = htmlspecialchars($row['username'] ?? '', ENT_QUOTES, 'UTF-8');
                $bname = htmlspecialchars($row['branch_name'] ?? '', ENT_QUOTES, 'UTF-8');
                $bid   = (int)$row['branch_id'];
                $count = (int)$row['letter_count'];
                $initial = strtoupper(mb_substr($uname, 0, 1));
              ?>
                <tr>
                  <td>
                    <div class="d-flex align-items-center">
                      <span class="rounded-circle d-inline-flex align-items-center justify-content-center mr-2"
                            style="width:34px;height:34px;background:rgba(37,99,235,.12);color:var(--brand);font-weight:700;">
                        <?php echo $initial; ?>
                      </span>
                      <span class="font-weight-semibold"><?php echo $uname; ?></span>
                    </div>
                  </td>
                  <td><?php echo $bname; ?></td>
                  <td class="text-center"><?php echo number_format($count); ?></td>
                  <td class="text-center">
                    <button class="btn btn-primary btn-sm view-details"
                            data-user-id="<?php echo $uid; ?>"
                            data-username="<?php echo $uname; ?>"
                            data-branch="<?php echo $bid; ?>"
                            data-toggle="modal"
                            data-target="#letterDetailsModal">
                      <i class="fas fa-eye"></i> View Details
                    </button>
                  </td>
                </tr>
              <?php endwhile; ?>
            </tbody>
          </table>
        </div>
      <?php endif; ?>
      <?php $stmt->close(); $conn->close(); ?>
    </div>
  </div>
</div>

<!-- Modal -->
<div id="letterDetailsModal" class="modal fade" tabindex="-1" role="dialog" aria-labelledby="letterDetailsModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-lg" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="letterDetailsModalLabel">
          <i class="fa fa-envelopes-bulk mr-2"></i>User's Letters
        </h5>
        <button type="button" class="close" data-dismiss="modal" aria-label="Close">
          <span>&times;</span>
        </button>
      </div>
      <div class="modal-body">
        <div id="letterDetailsContent">
          <div class="d-flex align-items-center justify-content-center py-5 text-secondary">
            <div class="spinner-border spinner-border-sm mr-2" role="status"></div>Loading letters…
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button class="btn btn-outline-secondary" data-dismiss="modal"><i class="fa fa-times mr-1"></i>Close</button>
      </div>
    </div>
  </div>
</div>

<!-- Scripts -->
<script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
<script src="https://cdn.datatables.net/1.10.21/js/jquery.dataTables.min.js"></script>
<script src="https://maxcdn.bootstrapcdn.com/bootstrap/4.0.0/js/bootstrap.min.js"></script>

<script>
  $(function() {
    $('#usersTable').DataTable({ pageLength: 25 });
    $(document).on('click', '.view-details', function() {
      var userId   = $(this).data('user-id');
      var branchId = $(this).data('branch');
      var uname    = $(this).data('username');

      $('#letterDetailsModalLabel').html('Letters for <strong>' + $('<div>').text(uname).html() + '</strong>');
      $('#letterDetailsContent').html('<div class="alert alert-info mb-0">Please wait, loading letters...</div>');

      $.ajax({
        url: 'get_user_letters.php',
        method: 'GET',
        data: { user_id: userId, branch_id: branchId },
        success: function(response) {
          $('#letterDetailsContent').html(response);
        },
        error: function() {
          $('#letterDetailsContent').html('<div class="alert alert-danger mb-0">Failed to load letters.</div>');
        }
      });
    });
  });
</script>
</body>
</html>
