<?php
// Start the session
session_start();

// Require login + role
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['staff_user','branch_head','letter_admin','super_admin'])) {
    header('Location: unauthorized.php'); exit();
}

// Include DB
require_once 'db_connection.php';

// Session context
$user_id   = (int)($_SESSION['user_id']   ?? 0);
$branch_id = (int)($_SESSION['branch_id'] ?? 0);
$role      = $_SESSION['role'] ?? 'user';

// Defaults (sticky)
$file_name = $file_number = $subject = $cupboard_name = $rack_name = $folder_name = $file_order_number = $status = "";
$error_message = "";

// --- Fetch username (prepared) ---
$username = "";
if ($user_id > 0) {
    $stmt = $conn->prepare("SELECT username FROM users WHERE user_id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $stmt->bind_result($username);
    $stmt->fetch();
    $stmt->close();
}

// --- Fetch branch name (prepared) ---
$branch_name = "";
if ($branch_id > 0) {
    $stmt = $conn->prepare("SELECT branch_name FROM branches WHERE branch_id = ?");
    $stmt->bind_param("i", $branch_id);
    $stmt->execute();
    $stmt->bind_result($branch_name);
    $stmt->fetch();
    $stmt->close();
}

// --- Handle form submission ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit'])) {
    // Trim input
    $file_name         = trim($_POST['file_name'] ?? "");
    $file_number       = trim($_POST['file_number'] ?? "");
    $subject           = trim($_POST['subject'] ?? "");
    $cupboard_name     = trim($_POST['cupboard_name'] ?? "");
    $rack_name         = trim($_POST['rack_name'] ?? "");
    $folder_name       = trim($_POST['folder_name'] ?? "");
    $file_order_number = trim($_POST['file_order_number'] ?? "");
    $status            = trim($_POST['status'] ?? "");

    // Basic validation
    $required = [$file_name,$file_number,$subject,$cupboard_name,$rack_name,$folder_name,$file_order_number,$status];
    if (in_array("", $required, true)) {
        $error_message = "All fields are required.";
    } elseif (!ctype_digit($file_order_number) || (int)$file_order_number < 0) {
        $error_message = "File order number must be a non-negative integer.";
    } elseif (!in_array($status, ['Active','Archived','Under Review'], true)) {
        $error_message = "Invalid status.";
    } else {
        // Optional: enforce unique file_number per branch
        $stmt = $conn->prepare("SELECT 1 FROM file_management WHERE branch_id = ? AND file_number = ? LIMIT 1");
        $stmt->bind_param("is", $branch_id, $file_number);
        $stmt->execute();
        $stmt->store_result();
        $exists = $stmt->num_rows > 0;
        $stmt->close();

        if ($exists) {
            $error_message = "A file with this number already exists in your branch.";
        } else {
            // Insert (prepared)
            $sql = "INSERT INTO file_management
                    (file_name, file_number, subject, branch_id, user_id, cupboard_name, rack_name, folder_name, file_order_number, status)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            $order = (int)$file_order_number;
            $stmt->bind_param(
                "sss iissssis",
                /* file_name   */ $file_name,
                /* file_number */ $file_number,
                /* subject     */ $subject,
                /* branch_id   */ $branch_id,
                /* user_id     */ $user_id,
                /* cupboard    */ $cupboard_name,
                /* rack        */ $rack_name,
                /* folder      */ $folder_name,
                /* order       */ $order,
                /* status      */ $status
            );
            // Note: spaces in types above are for readability; PHP ignores them.
            $types = "sss iissssis"; // keep comment in sync only
            // Actually bind again correctly (no spaces)
            $stmt->bind_param(
                "sssiisssis",
                $file_name, $file_number, $subject, $branch_id, $user_id,
                $cupboard_name, $rack_name, $folder_name, $order, $status
            );

            if ($stmt->execute()) {
                $_SESSION['success_message'] = "File \"".htmlspecialchars($file_name, ENT_QUOTES, 'UTF-8')."\" created successfully.";
                // Clear sticky values after success
                $file_name = $file_number = $subject = $cupboard_name = $rack_name = $folder_name = $file_order_number = $status = "";
                header("Location: file_entry.php"); // PRG pattern to avoid resubmit on refresh
                exit();
            } else {
                $_SESSION['error_message'] = "Database error: ".$stmt->error;
            }
            $stmt->close();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>File Entry</title>

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
        rel="stylesheet"
        integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
        crossorigin="anonymous"/>

  <style>
    :root { --bg:#f6f8fb; }
    body { background: var(--bg); }
    .wrap { max-width: 980px; }
    .card { border: none; border-radius: 1rem; box-shadow: 0 10px 30px rgba(0,0,0,.06); }
    .section-title { font-weight: 800; margin-bottom: .5rem; }
  </style>
</head>
<body>
<div class="container py-4 wrap">

  <div class="card mb-3">
    <div class="card-body">
      <h4 class="section-title">File Entry Form</h4>
      <p class="text-muted mb-0">Create a new file record for <strong><?= htmlspecialchars($branch_name ?: '—', ENT_QUOTES, 'UTF-8'); ?></strong></p>
    </div>
  </div>

  <?php if (!empty($_SESSION['success_message'])): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
      <i class="bi bi-check-circle me-1"></i>
      <?= $_SESSION['success_message']; unset($_SESSION['success_message']); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
  <?php endif; ?>

  <?php if (!empty($_SESSION['error_message'])): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
      <?= $_SESSION['error_message']; unset($_SESSION['error_message']); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
  <?php endif; ?>

  <?php if (!empty($error_message)): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
      <?= htmlspecialchars($error_message, ENT_QUOTES, 'UTF-8'); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
  <?php endif; ?>

  <form action="file_entry.php" method="POST" novalidate class="needs-validation">
    <div class="card mb-3">
      <div class="card-body">
        <h6 class="text-muted mb-3">User & Branch</h6>
        <div class="row g-3">
          <div class="col-md-6">
            <label class="form-label">Username</label>
            <input type="text" class="form-control" value="<?= htmlspecialchars($username ?: '—', ENT_QUOTES, 'UTF-8'); ?>" readonly>
          </div>
          <div class="col-md-6">
            <label class="form-label">Branch</label>
            <input type="text" class="form-control" value="<?= htmlspecialchars($branch_name ?: '—', ENT_QUOTES, 'UTF-8'); ?>" readonly>
          </div>
        </div>
      </div>
    </div>

    <div class="card mb-3">
      <div class="card-body">
        <h6 class="text-muted mb-3">File Information</h6>
        <div class="row g-3">
          <div class="col-md-6">
            <label for="file_name" class="form-label">File Name</label>
            <input id="file_name" name="file_name" type="text" class="form-control" required
                   value="<?= htmlspecialchars($file_name, ENT_QUOTES, 'UTF-8'); ?>">
            <div class="invalid-feedback">Please enter file name.</div>
          </div>
          <div class="col-md-6">
            <label for="file_number" class="form-label">File Number</label>
            <input id="file_number" name="file_number" type="text" class="form-control" required
                   pattern="[\w\-/\. ]{1,50}"
                   value="<?= htmlspecialchars($file_number, ENT_QUOTES, 'UTF-8'); ?>">
            <div class="invalid-feedback">Please enter a valid file number.</div>
          </div>

          <div class="col-12">
            <label for="subject" class="form-label">Subject</label>
            <textarea id="subject" name="subject" class="form-control" rows="3" required><?= htmlspecialchars($subject, ENT_QUOTES, 'UTF-8'); ?></textarea>
            <div class="invalid-feedback">Please enter subject.</div>
          </div>

          <div class="col-md-4">
            <label for="cupboard_name" class="form-label">Cupboard</label>
            <input id="cupboard_name" name="cupboard_name" type="text" class="form-control" required
                   value="<?= htmlspecialchars($cupboard_name, ENT_QUOTES, 'UTF-8'); ?>">
            <div class="invalid-feedback">Required.</div>
          </div>
          <div class="col-md-4">
            <label for="rack_name" class="form-label">Rack</label>
            <input id="rack_name" name="rack_name" type="text" class="form-control" required
                   value="<?= htmlspecialchars($rack_name, ENT_QUOTES, 'UTF-8'); ?>">
            <div class="invalid-feedback">Required.</div>
          </div>
          <div class="col-md-4">
            <label for="folder_name" class="form-label">Folder</label>
            <input id="folder_name" name="folder_name" type="text" class="form-control" required
                   value="<?= htmlspecialchars($folder_name, ENT_QUOTES, 'UTF-8'); ?>">
            <div class="invalid-feedback">Required.</div>
          </div>

          <div class="col-md-6">
            <label for="file_order_number" class="form-label">File Order #</label>
            <input id="file_order_number" name="file_order_number" type="number" min="0" step="1" class="form-control" required
                   value="<?= htmlspecialchars($file_order_number, ENT_QUOTES, 'UTF-8'); ?>">
            <div class="invalid-feedback">Enter a non-negative number.</div>
          </div>

          <div class="col-md-6">
            <label for="status" class="form-label">Status</label>
            <select id="status" name="status" class="form-select" required>
              <?php
                $opts = ['Active','Archived','Under Review'];
                foreach ($opts as $opt) {
                    $sel = ($status === $opt) ? 'selected' : '';
                    echo '<option value="'.htmlspecialchars($opt, ENT_QUOTES, 'UTF-8').'" '.$sel.'>'.$opt.'</option>';
                }
              ?>
            </select>
            <div class="invalid-feedback">Please choose a status.</div>
          </div>
        </div>
      </div>
    </div>

    <div class="d-flex gap-2">
      <button type="submit" name="submit" class="btn btn-success">
        <i class="fa-solid fa-floppy-disk me-1"></i> Submit
      </button>
      <?php
        if     ($role === 'staff_user')   $back = 'staff_user_dashboard.php';
        elseif ($role === 'branch_head')  $back = 'branch_head_dashboard.php';
        elseif ($role === 'letter_admin') $back = 'letter_admin_dashboard.php';
        else                              $back = 'super_admin_dashboard.php';
      ?>
      <a href="<?= $back; ?>" class="btn btn-secondary">
        <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
      </a>
    </div>
  </form>

</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
        integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
        crossorigin="anonymous"></script>
<script>
// Bootstrap validation
(() => {
  const forms = document.querySelectorAll('.needs-validation');
  Array.from(forms).forEach(form => {
    form.addEventListener('submit', event => {
      if (!form.checkValidity()) {
        event.preventDefault(); event.stopPropagation();
      }
      form.classList.add('was-validated');
    }, false);
  });
})();
</script>
</body>
</html>
<?php
$conn->close();
