<?php
session_start();
require_once 'db_connection.php';

// Allow only letter_admin (adjust if others may access)
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'letter_admin') {
    header('Location: unauthorized.php');
    exit();
}

// Init
$letter_number = $letter_date = $department = $subject = '';
$branch_id = $role = $hod_comment = $priority = '';
$user_id = '';
$error_message = $success_message = '';

// Fetch branches (for dropdown)
$branches = $conn->query("SELECT branch_id, branch_name FROM branches");

// Handle POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Gather & normalize
    $letter_number = trim($_POST['letter_number'] ?? '');
    $letter_date   = trim($_POST['letter_date'] ?? '');
    $department    = trim($_POST['department'] ?? '');
    $subject       = trim($_POST['subject'] ?? '');
    $branch_id     = (int)($_POST['branch_id'] ?? 0);
    $role          = trim($_POST['role'] ?? 'branch_head'); // only branch_head in UI
    $user_id       = (int)($_POST['user_id'] ?? 0);
    $hod_comment   = trim($_POST['hod_comment'] ?? '');
    $priority      = trim($_POST['priority'] ?? 'Low');

    // Basic validation
    if ($letter_number === '' || $letter_date === '' || $department === '' || $subject === '' ||
        $branch_id <= 0 || $role === '' || $user_id <= 0 || $hod_comment === '' || $priority === '') {
        $error_message = "All fields are required.";
    }

    // File upload (optional)
    $scan_copy = null;
    if (empty($error_message) && isset($_FILES['scan_copy']) && $_FILES['scan_copy']['error'] !== UPLOAD_ERR_NO_FILE) {
        $fileErr = $_FILES['scan_copy']['error'];
        if ($fileErr === UPLOAD_ERR_OK) {
            $tmp   = $_FILES['scan_copy']['tmp_name'];
            $name  = $_FILES['scan_copy']['name'];
            $size  = (int)$_FILES['scan_copy']['size'];

            // Constraints: 5MB limit, pdf/jpg/jpeg/png
            $maxSize = 5 * 1024 * 1024;
            if ($size > $maxSize) {
                $error_message = "File too large. Maximum allowed size is 5MB.";
            } else {
                // Validate MIME using finfo
                $finfo = new finfo(FILEINFO_MIME_TYPE);
                $mime  = $finfo->file($tmp);
                $allowed = [
                    'application/pdf' => 'pdf',
                    'image/jpeg'      => 'jpg',
                    'image/png'       => 'png'
                ];
                if (!isset($allowed[$mime])) {
                    $error_message = "Invalid file type. Only PDF, JPG, and PNG are allowed.";
                } else {
                    // Ensure uploads directory exists
                    $uploadDir = __DIR__ . '/uploads';
                    if (!is_dir($uploadDir)) {
                        @mkdir($uploadDir, 0755, true);
                    }
                    // Unique filename
                    $ext = $allowed[$mime];
                    $slug = preg_replace('/[^a-zA-Z0-9_-]+/', '_', pathinfo($name, PATHINFO_FILENAME));
                    $unique = $slug . '_' . date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . '.' . $ext;

                    $destFs = $uploadDir . '/' . $unique;      // filesystem path
                    $destDb = 'uploads/' . $unique;            // path stored in DB (web-accessible)

                    if (!@move_uploaded_file($tmp, $destFs)) {
                        $error_message = "Error saving the uploaded file.";
                    } else {
                        $scan_copy = $destDb;
                    }
                }
            }
        } else {
            $error_message = "Upload error (code: $fileErr).";
        }
    }

    // Insert
    if (empty($error_message)) {
        $sql = "INSERT INTO letters
                (letter_number, letter_date, department, subject, branch_id, role, user_id, scan_copy, hod_comment, priority)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        if ($stmt = $conn->prepare($sql)) {
            // Types: s s s s i s i s s s   (10 params)
            $stmt->bind_param(
                "ssssisisss",
                $letter_number,
                $letter_date,
                $department,
                $subject,
                $branch_id,
                $role,
                $user_id,
                $scan_copy,     // can be NULL, column should allow NULL
                $hod_comment,
                $priority
            );
            if ($stmt->execute()) {
                $success_message = "Letter added successfully!";
                // Reset form values
                $letter_number = $letter_date = $department = $subject = '';
                $branch_id = $role = $hod_comment = $priority = '';
                $user_id = '';
            } else {
                $error_message = "Database error: " . $stmt->error;
            }
            $stmt->close();
        } else {
            $error_message = "Error preparing query: " . $conn->error;
        }
    }
}
// No $conn->close() here if you still need it later on this request
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Enter Letter – Letter Admin</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
        rel="stylesheet"
        integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
        crossorigin="anonymous">
  <!-- Select2 -->
  <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet"/>
  <!-- Font Awesome -->
  <link rel="stylesheet"
        href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css"
        referrerpolicy="no-referrer"/>

  <style>
    :root{
      --brand:#0d6efd; --brand-2:#5aa2ff; --bg:#f6f8fb;
    }
    body{ background:var(--bg); }
    .appbar{
      background: linear-gradient(135deg, var(--brand), var(--brand-2));
      color:#fff; padding:.9rem 1rem; display:flex; justify-content:space-between; align-items:center;
      box-shadow: 0 6px 24px rgba(13,110,253,.25); position:sticky; top:0; z-index:1020;
      margin-bottom:1rem;
    }
    .logo-square{ width:44px; height:44px; object-fit:cover; border-radius:.6rem; border:2px solid rgba(255,255,255,.35); }
    .card-elevated{ border:none; border-radius:1rem; box-shadow:0 10px 30px rgba(0,0,0,.08); }
    .select2-container .select2-selection--single{
      height: calc(2.5rem + 2px);
      padding: .5rem .75rem;
      border: 1px solid #ced4da; border-radius:.5rem;
      display:flex; align-items:center;
    }
    .required:after{ content:" *"; color:#dc3545; }
  </style>
</head>
<body>

<!-- Header -->
<div class="appbar">
  <div class="d-flex align-items-center gap-3">
    <img src="logo1.png" alt="Left Logo" class="logo-square" onerror="this.style.display='none'">
    <div>
      <h5 class="mb-0 fw-bold">Office Management System</h5>
      <small class="text-white-75">Valvettithurai Urban Council</small>
    </div>
  </div>
  <div class="d-flex align-items-center gap-2">
    <a href="letter_admin_dashboard.php" class="btn btn-light btn-sm">
      <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
    </a>
  </div>
</div>

<div class="container py-2">

  <div class="mb-3">
    <h3 class="fw-bold mb-1">Enter Letter</h3>
    <div class="text-muted">Create a new incoming letter and assign it to the Branch Head.</div>
  </div>

  <?php if (!empty($success_message)): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
      <i class="fa-solid fa-circle-check me-1"></i> <?= htmlspecialchars($success_message, ENT_QUOTES, 'UTF-8'); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
  <?php endif; ?>

  <?php if (!empty($error_message)): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
      <i class="fa-solid fa-triangle-exclamation me-1"></i> <?= htmlspecialchars($error_message, ENT_QUOTES, 'UTF-8'); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
  <?php endif; ?>

  <div class="card card-elevated">
    <div class="card-body">
      <form method="POST" enctype="multipart/form-data" class="row g-3">

        <!-- Left column -->
        <div class="col-12 col-lg-6">
          <div class="mb-3">
            <label class="form-label required">Letter Number</label>
            <input type="text" name="letter_number" class="form-control" value="<?= htmlspecialchars($letter_number, ENT_QUOTES, 'UTF-8'); ?>" required>
          </div>

          <div class="mb-3">
            <label class="form-label required">Letter Date</label>
            <input type="date" name="letter_date" class="form-control" value="<?= htmlspecialchars($letter_date, ENT_QUOTES, 'UTF-8'); ?>" required>
          </div>

          <div class="mb-3">
            <label class="form-label required">Department</label>
            <input type="text" name="department" class="form-control" value="<?= htmlspecialchars($department, ENT_QUOTES, 'UTF-8'); ?>" required>
          </div>

          <div class="mb-3">
            <label class="form-label required">Subject</label>
            <input type="text" name="subject" class="form-control" value="<?= htmlspecialchars($subject, ENT_QUOTES, 'UTF-8'); ?>" required>
          </div>

          <div class="mb-3">
            <label class="form-label required">Branch</label>
            <select name="branch_id" id="branch_id" class="form-select" required>
              <option value="">Select Branch</option>
              <?php if ($branches && $branches->num_rows > 0): ?>
                <?php while ($b = $branches->fetch_assoc()): ?>
                  <option value="<?= (int)$b['branch_id']; ?>" <?= ((string)$branch_id === (string)$b['branch_id']) ? 'selected' : ''; ?>>
                    <?= htmlspecialchars($b['branch_name'], ENT_QUOTES, 'UTF-8'); ?>
                  </option>
                <?php endwhile; ?>
              <?php endif; ?>
            </select>
          </div>
        </div>

        <!-- Right column -->
        <div class="col-12 col-lg-6">
          <div class="mb-3">
            <label class="form-label required">Assign To Role</label>
            <select name="role" class="form-select" required>
              <option value="branch_head" <?= ($role === 'branch_head' ? 'selected' : ''); ?>>Branch Head</option>
            </select>
          </div>

          <div class="mb-3">
            <label class="form-label required">Assign To (Branch Head)</label>
            <select name="user_id" id="user_id" class="form-select" required>
              <option value="">Select User</option>
              <!-- populated by AJAX on branch selection -->
            </select>
          </div>

          <div class="mb-3">
            <label class="form-label">Scan Copy (PDF/JPG/PNG, ≤ 5MB)</label>
            <input type="file" name="scan_copy" class="form-control" accept=".pdf,.jpg,.jpeg,.png">
          </div>

          <div class="mb-3">
            <label class="form-label required">HOD Comment</label>
            <textarea name="hod_comment" class="form-control" rows="3" required><?= htmlspecialchars($hod_comment, ENT_QUOTES, 'UTF-8'); ?></textarea>
          </div>

          <div class="mb-3">
            <label class="form-label required">Priority</label>
            <select name="priority" class="form-select" required>
              <option value="Low"    <?= ($priority === 'Low'    ? 'selected' : ''); ?>>Low</option>
              <option value="Medium" <?= ($priority === 'Medium' ? 'selected' : ''); ?>>Medium</option>
              <option value="High"   <?= ($priority === 'High'   ? 'selected' : ''); ?>>High</option>
            </select>
          </div>
        </div>

        <div class="col-12">
          <button type="submit" class="btn btn-success">
            <i class="fa-solid fa-floppy-disk me-1"></i> Save Letter
          </button>
          <a href="letter_admin_dashboard.php" class="btn btn-outline-secondary ms-2">
            <i class="fa-solid fa-arrow-left-long me-1"></i> Back
          </a>
        </div>

      </form>
    </div>
  </div>

</div>

<!-- Scripts -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
        integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
        crossorigin="anonymous"></script>
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

<script>
  $(function(){
    // Enhance selects
    $('#user_id').select2({ placeholder: "Select User", width: '100%' });

    // Load users for selected branch
    function loadBranchUsers(branchId, selectedUserId) {
      if (!branchId) {
        $('#user_id').html('<option value="">Select User</option>').trigger('change');
        return;
      }
      $.get('get_users.php', { branch_id: branchId }, function (html) {
        $('#user_id').html(html);
        if (selectedUserId) {
          $('#user_id').val(selectedUserId).trigger('change');
        } else {
          $('#user_id').val(null).trigger('change');
        }
      });
    }

    // On branch change, fetch branch head users
    $('#branch_id').on('change', function(){
      loadBranchUsers($(this).val(), null);
    });

    // If branch was preselected, load users (for validation error redisplay)
    const presetBranch = $('#branch_id').val();
    <?php if (!empty($branch_id)): ?>
      loadBranchUsers('<?= (int)$branch_id; ?>', '<?= (int)$user_id; ?>');
    <?php endif; ?>
  });
</script>
</body>
</html>
