<?php
// edit_user.php (secure & polished)
session_start();
require_once 'db_connection.php';

/* ---------- CSRF token ---------- */
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf = $_SESSION['csrf_token'];

/* ---------- Helpers ---------- */
function h($s){ return htmlspecialchars($s ?? '', ENT_QUOTES, 'UTF-8'); }
function valid_role($r){ return in_array($r, ['letter_admin','branch_head','staff_user','super_admin'], true); }
function valid_status($s){ return in_array($s, ['active','inactive'], true); }

/* ---------- Fetch branches for dropdown ---------- */
$branches = [];
if ($stmt = $conn->prepare("SELECT branch_id, branch_name FROM branches ORDER BY branch_name ASC")) {
    $stmt->execute();
    $res = $stmt->get_result();
    while ($row = $res->fetch_assoc()) { $branches[] = $row; }
    $stmt->close();
}

/* ---------- Load user (GET) ---------- */
$user = null;
$errors = [];
$success = '';

if (!isset($_GET['user_id']) && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    $errors[] = "Missing user_id.";
} 

// On POST we’ll use hidden user_id; on GET we use query param
$user_id = null;
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['user_id'])) {
    $user_id = (int)$_GET['user_id'];
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['user_id'])) {
    $user_id = (int)$_POST['user_id'];
}

if ($user_id) {
    if ($stmt = $conn->prepare("SELECT user_id, username, branch_id, role, status FROM users WHERE user_id = ?")) {
        $stmt->bind_param('i', $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        if ($res->num_rows === 0) {
            $errors[] = "User not found.";
        } else {
            $user = $res->fetch_assoc();
        }
        $stmt->close();
    } else {
        $errors[] = "Unable to load user.";
    }
}

/* ---------- Handle POST (update) ---------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !$errors) {
    // CSRF
    if (!isset($_POST['csrf']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf'])) {
        $errors[] = "Invalid form submission. Please try again.";
    }

    $username  = trim($_POST['username'] ?? '');
    $role      = $_POST['role'] ?? '';
    $status    = $_POST['status'] ?? '';
    $branch_id_new = $_POST['branch_id'] ?? '';
    $password_raw  = $_POST['password'] ?? '';

    // Validate inputs
    if ($username === '' || strlen($username) < 3) $errors[] = "Username must be at least 3 characters.";
    if (!valid_role($role)) $errors[] = "Invalid role selected.";
    if (!valid_status($status)) $errors[] = "Invalid status selected.";
    if ($branch_id_new === '') $errors[] = "Please select a branch.";

    // Ensure branch exists
    if (!$errors) {
        if ($stmt = $conn->prepare("SELECT 1 FROM branches WHERE branch_id = ?")) {
            $stmt->bind_param('s', $branch_id_new);
            $stmt->execute();
            $stmt->store_result();
            if ($stmt->num_rows === 0) {
                $errors[] = "Selected branch does not exist.";
            }
            $stmt->close();
        } else {
            $errors[] = "Unable to verify branch.";
        }
    }

    // Ensure username uniqueness (excluding current user)
    if (!$errors) {
        if ($stmt = $conn->prepare("SELECT 1 FROM users WHERE username = ? AND user_id <> ?")) {
            $stmt->bind_param('si', $username, $user_id);
            $stmt->execute();
            $stmt->store_result();
            if ($stmt->num_rows > 0) {
                $errors[] = "Username already in use by another account.";
            }
            $stmt->close();
        } else {
            $errors[] = "Unable to validate username.";
        }
    }

    // Proceed if valid
    if (!$errors) {
        $conn->begin_transaction();
        try {
            // Build update for users table
            if ($password_raw !== '') {
                // Hash new password
                $password_hashed = password_hash($password_raw, PASSWORD_DEFAULT);
                $sql = "UPDATE users SET username = ?, role = ?, status = ?, password = ?, branch_id = ? WHERE user_id = ?";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param('sssssi', $username, $role, $status, $password_hashed, $branch_id_new, $user_id);
            } else {
                $sql = "UPDATE users SET username = ?, role = ?, status = ?, branch_id = ? WHERE user_id = ?";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param('ssssi', $username, $role, $status, $branch_id_new, $user_id);
            }
            if (!$stmt->execute()) {
                throw new Exception("Failed to update user.");
            }
            $stmt->close();

            // If branch changed, optionally update related records
            $old_branch_id = $user['branch_id'];
            if ((string)$old_branch_id !== (string)$branch_id_new) {
                // Update file_management
                if ($stmt = $conn->prepare("UPDATE file_management SET branch_id = ? WHERE user_id = ?")) {
                    $stmt->bind_param('si', $branch_id_new, $user_id);
                    if (!$stmt->execute()) throw new Exception("Failed to update file records.");
                    $stmt->close();
                }
                // Update letters
                if ($stmt = $conn->prepare("UPDATE letters SET branch_id = ? WHERE user_id = ?")) {
                    $stmt->bind_param('si', $branch_id_new, $user_id);
                    if (!$stmt->execute()) throw new Exception("Failed to update letter records.");
                    $stmt->close();
                }
            }

            $conn->commit();
            $success = "User updated successfully.";
            // Refresh $user for display
            if ($stmt = $conn->prepare("SELECT user_id, username, branch_id, role, status FROM users WHERE user_id = ?")) {
                $stmt->bind_param('i', $user_id);
                $stmt->execute();
                $user = $stmt->get_result()->fetch_assoc();
                $stmt->close();
            }
            // Rotate CSRF after successful POST
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
            $csrf = $_SESSION['csrf_token'];

        } catch (Exception $e) {
            $conn->rollback();
            $errors[] = $e->getMessage();
        }
    }
}

?>
<!doctype html>
<html lang="en" data-bs-theme="light">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Edit User · Office Management System</title>

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body{
      min-height:100vh;
      background:#f4f7fb;
      display:flex; align-items:center; justify-content:center;
      padding: 24px 12px;
    }
    .card{
      width:100%;
      max-width: 640px;
      border:0; border-radius:16px;
      box-shadow: 0 12px 28px rgba(0,0,0,.12);
      overflow:hidden;
    }
    .card-header{
      background:#fff; border-bottom:1px solid rgba(0,0,0,.06);
    }
  </style>
</head>
<body>
<div class="card">
  <div class="card-header">
    <div class="d-flex align-items-center justify-content-between">
      <h1 class="h5 m-0">Edit User</h1>
      <a href="view_users.php" class="btn btn-sm btn-outline-secondary">← Back to Users</a>
    </div>
  </div>
  <div class="card-body">
    <?php if ($success): ?>
      <div class="alert alert-success d-flex align-items-center" role="alert">
        <div class="me-2">✅</div>
        <div><?= h($success); ?></div>
      </div>
    <?php endif; ?>

    <?php if ($errors): ?>
      <div class="alert alert-danger" role="alert">
        <strong>We couldn’t save your changes:</strong>
        <ul class="mb-0">
          <?php foreach ($errors as $e): ?>
            <li><?= h($e); ?></li>
          <?php endforeach; ?>
        </ul>
      </div>
    <?php endif; ?>

    <?php if ($user): ?>
      <form method="POST" novalidate>
        <input type="hidden" name="csrf" value="<?= h($csrf); ?>">
        <input type="hidden" name="user_id" value="<?= (int)$user['user_id']; ?>">

        <div class="mb-3">
          <label for="username" class="form-label">Username <span class="text-danger">*</span></label>
          <input type="text" id="username" name="username" class="form-control" value="<?= h($user['username']); ?>" required minlength="3" autocomplete="username">
        </div>

        <div class="row g-3">
          <div class="col-sm-6">
            <label for="role" class="form-label">Role <span class="text-danger">*</span></label>
            <select id="role" name="role" class="form-select" required>
              <?php
                $roleOptions = [
                  'branch_head'  => 'Branch Head',
                  'staff_user'   => 'Staff User',
                  'letter_admin' => 'Letter Admin',
                  'super_admin'  => 'Super Admin'
                ];
                foreach ($roleOptions as $val=>$label):
              ?>
                <option value="<?= h($val); ?>" <?= ($user['role']===$val?'selected':''); ?>><?= h($label); ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-sm-6">
            <label for="status" class="form-label">Status <span class="text-danger">*</span></label>
            <select id="status" name="status" class="form-select" required>
              <option value="active"   <?= ($user['status']==='active'?'selected':''); ?>>Active</option>
              <option value="inactive" <?= ($user['status']==='inactive'?'selected':''); ?>>Inactive</option>
            </select>
          </div>
        </div>

        <div class="mb-3 mt-3">
          <label for="branch_id" class="form-label">Branch <span class="text-danger">*</span></label>
          <select id="branch_id" name="branch_id" class="form-select" required>
            <option value="">Select Branch</option>
            <?php foreach ($branches as $b): ?>
              <option value="<?= h($b['branch_id']); ?>" <?= ((string)$user['branch_id']===(string)$b['branch_id']?'selected':''); ?>>
                <?= h($b['branch_name']); ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="mb-3">
          <label for="password" class="form-label">New Password (leave blank to keep current)</label>
          <div class="input-group">
            <input type="password" id="password" name="password" class="form-control" autocomplete="new-password" minlength="6">
            <button class="btn btn-outline-secondary" type="button" id="togglePwd">Show</button>
          </div>
          <div class="form-text">Minimum 6 characters recommended.</div>
        </div>

        <div class="d-flex gap-2">
          <button class="btn btn-primary" type="submit">
            Save Changes
          </button>
          <a class="btn btn-outline-secondary" href="view_users.php">Cancel</a>
        </div>
      </form>
    <?php else: ?>
      <div class="text-center">
        <p class="text-muted mb-3">Nothing to edit.</p>
        <a class="btn btn-outline-secondary" href="view_users.php">Back to Users</a>
      </div>
    <?php endif; ?>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Show/Hide Password
  const btn = document.getElementById('togglePwd');
  const pwd = document.getElementById('password');
  btn?.addEventListener('click', ()=>{
    const isPwd = pwd.getAttribute('type') === 'password';
    pwd.setAttribute('type', isPwd ? 'text' : 'password');
    btn.textContent = isPwd ? 'Hide' : 'Show';
  });
</script>
</body>
</html>
